/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/***************************************************************************************************************/
/* TABLE: Population Diversity and the Frequency of Civil Conflict Onset across Countries - Mediating Channels */
/***************************************************************************************************************/

# delimit ;
clear all ;

set more off ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/mtab_mechs.log", text replace ;

/*****************************************/
/* Load the AAGK cross-sectional dataset */
/*****************************************/

use "../../data/aagk_xcountry.dta", clear ;

/**************************************/
/* Apply relevant log transformations */
/**************************************/

foreach var of varlist num_ccst_60_17_avg_pri1 numgrps_fea03 { ;
  gen ln`var' = ln(1 + `var') ;
} ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variable and the mediating channels */
label variable pdiv_aa          "Population diversity (ancestry adjusted)" ;
label variable lnnumgrps_fea03  "Log number of ethnic groups" ;
label variable wvs_trust        "Prevalence of interpersonal trust" ;
label variable wvs_leftright_sd "Variation in political attitudes" ;

/* Geographical variables */
label variable abslat  "Absolute latitude" ;
label variable ruggavg "Ruggedness" ;
label variable elevavg "Mean elevation" ;
label variable elevrng "Range of elevation" ;
label variable suitavg "Mean land suitability" ;
label variable suitrng "Range of land suitability" ;
label variable distcr  "Distance to nearest waterway" ;
label variable island  "Island nation dummy" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global contall "africa asia namerica samerica oceania" ;
global regions "wb_ssa wb_lac" ;

/******************************************************/
/* Generate the sample indicators for the regressions */
/******************************************************/

qui egen smpl_flag1 = rowmiss(lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} lnnumgrps_fea03) ;
qui egen smpl_flag2 = rowmiss(lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} wvs_trust) ;
qui egen smpl_flag3 = rowmiss(lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} wvs_leftright_sd) ;

qui replace smpl_flag1 = (smpl_flag1 == 0) ;
qui replace smpl_flag2 = (smpl_flag2 == 0) ;
qui replace smpl_flag3 = (smpl_flag3 == 0) ;

/* For regressions incorporating WVS measures, ignore "Philippines" and "Dominican Republic" since WVS measures are noisy for these observations */
replace smpl_flag2 = 0 if wbcode == "PHL" | wbcode == "DOM" ;
replace smpl_flag3 = 0 if wbcode == "PHL" | wbcode == "DOM" ;

/*************************************************************************************************************************/
/* Standardize the WVS measure of heterogeneity in political attitudes since it is based on an ordinal response variable */
/*************************************************************************************************************************/

egen std_wvs_polhet = std(wvs_leftright_sd) if smpl_flag3 == 1 ;
label variable std_wvs_polhet "Variation in political attitudes" ;

/***********************/
/* Run the regressions */
/***********************/

/* ------------------------------------------ */
/* Mediating channel: Number of ethnic groups */
/* ------------------------------------------ */

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: reg lnnumgrps_fea03 pdiv_aa ${geovars} ${contall} if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum numgrps_fea03 if e(sample) == 1 ;
local avggrps = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col1_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avggrps') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnumgrps_fea03 pdiv_aa ${geovars} ${contall} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col1 ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${contall} if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col2_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${contall} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col2 ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa lnnumgrps_fea03 ${geovars} ${contall} if smpl_flag1 == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col3_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa lnnumgrps_fea03 ${geovars} ${contall} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col3 ;

/* -------------------------------------- */
/* Mediating channel: Interpersonal trust */
/* -------------------------------------- */

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: reg wvs_trust pdiv_aa ${geovars} ${regions} if smpl_flag2 == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col4_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr wvs_trust pdiv_aa ${geovars} ${regions} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col4 ;

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${regions} if smpl_flag2 == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col5_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${regions} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col5 ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa wvs_trust ${geovars} ${regions} if smpl_flag2 == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col6_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa wvs_trust ${geovars} ${regions} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col6 ;

/* --------------------------------------------------- */
/* Mediating channel: Variation in political attitudes */
/* --------------------------------------------------- */

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: reg std_wvs_polhet pdiv_aa ${geovars} ${regions} if smpl_flag3 == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col7_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr std_wvs_polhet pdiv_aa ${geovars} ${regions} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col7 ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${regions} if smpl_flag3 == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col8_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${regions} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col8 ;

/* -------- */
/* COLUMN 9 */
/* -------- */
eststo col9: reg lnnum_ccst_60_17_avg_pri1 pdiv_aa std_wvs_polhet ${geovars} ${regions} if smpl_flag3 == 1, vce(robust) noomitted vsquish ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
qui sum num_ccst_60_17_avg_pri1 if e(sample) == 1 ;
local avgconf = r(mean) ;
_pctile pdiv_aa if e(sample) == 1, percentiles(10 90) ;
eststo col9_me: nlcom (me_1090: _b[pdiv_aa] * (1 + `avgconf') * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr lnnum_ccst_60_17_avg_pri1 pdiv_aa std_wvs_polhet ${geovars} ${regions} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col9 ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 col9 using "../../results/tables/mtab_mechs.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(pdiv_aa lnnumgrps_fea03 wvs_trust std_wvs_polhet, relax) 
  order(pdiv_aa lnnumgrps_fea03 wvs_trust std_wvs_polhet) 
  indicate("Continent/region dummies=${contall} ${regions}" "Controls for geography=${geovars}", labels("\$\times\$" " ")) 
  stats(N pc_pdiv r2_a, fmt(%9.0f a2 a2) labels("Observations" "Partial \$R^2\$ of population diversity" "Adjusted \$R^2\$") layout(@ @ @)) 
  varwidth(44) msign("\$-\$") label 
  prehead("\begin{tabular*}{735pt}{@{\extracolsep{\fill}}lccccccccc}" 
          "\toprule" 
          "Mediating channel:&\multicolumn{3}{c}{Cultural fragmentation}&\multicolumn{3}{c}{Interpersonal trust}&\multicolumn{3}{c}{Preference heterogeneity}\\" 
          "\cmidrule(r){2-4}\cmidrule(lr){5-7}\cmidrule(l){8-10}") 
  numbers mlabels("OLS" "OLS" "OLS" "OLS" "OLS" "OLS" "OLS" "OLS" "OLS") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{1}{c}{Log number}&\multicolumn{2}{c}{Annual frequency}     &\multicolumn{1}{c}{Prevalence of}&\multicolumn{2}{c}{Annual frequency}     &\multicolumn{1}{c}{Variation}   &\multicolumn{2}{c}{Annual frequency}     \\" 
           "&\multicolumn{1}{c}{of ethnic} &\multicolumn{2}{c}{of new civil conflict}&\multicolumn{1}{c}{interpersonal}&\multicolumn{2}{c}{of new civil conflict}&\multicolumn{1}{c}{in political}&\multicolumn{2}{c}{of new civil conflict}\\" 
           "&\multicolumn{1}{c}{groups}    &\multicolumn{2}{c}{onsets, 1960--2017}   &\multicolumn{1}{c}{trust}        &\multicolumn{2}{c}{onsets, 1960--2017}   &\multicolumn{1}{c}{attitudes}   &\multicolumn{2}{c}{onsets, 1960--2017}   \\" 
           "\cmidrule(r){2-2}\cmidrule(lr){3-4}\cmidrule(lr){5-5}\cmidrule(lr){6-7}\cmidrule(lr){8-8}\cmidrule(l){9-10}") 
  varlabels(,elist(std_wvs_polhet \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1_me col2_me col3_me col4_me col5_me col6_me col7_me col8_me col9_me using "../../results/tables/mtab_mechs.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) 
  varwidth(44) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(me_1090 "Effect of 10th--90th \%ile move in diversity") 
  postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
