/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/***********************************************************************************************************************************************************/
/* TABLE: Population Diversity and the Frequency of Civil Conflict Onset across Countries - Robustness to Accounting for Spatial Autocorrelation in Errors */
/***********************************************************************************************************************************************************/

# delimit ;
clear all ;

set more off ;

/**********************************************************/
/* Define wrapper programs for running Conley regressions */
/**********************************************************/

/* Define a wrapper program around "x_ols.ado" that implements Conley's (1999) OLS estimation with spatial standard errors */
capture program drop conleyreg ;
program define conleyreg, eclass ;
  version 14.2 ;
  syntax varlist [if] ;

  tokenize `varlist' ;
  local lhs_var `1' ;
  local rhs_var `2' ;
  macro shift    2  ;
  local ctl_lst `*' ;

  local rhs_lst `"`rhs_var' `ctl_lst' _cons"' ;
  local reg_num : list sizeof rhs_lst ;

  quietly { ;
    preserve ;
    keep `if' ;
    x_ols conley_coord1 conley_coord2 conley_cutoff1 conley_cutoff2 `lhs_var' `rhs_var' `ctl_lst' conley_const, xreg(`reg_num') coord(2) ;
    restore ;
  } ;

  tempname b V ;
  matrix `b' = e(b) ;
  matrix `V' = cov_dep ;

  matrix drop cov_dep ;
  scalar drop fix ;

  matrix colnames `b' = `rhs_lst' ;
  matrix rownames `b' = `lhs_var' ;
  matrix colnames `V' = `rhs_lst' ;
  matrix rownames `V' = `rhs_lst' ;

  local N    = e(N) ;
  local r2   = e(r2) ;
  local r2_a = e(r2_a) ;

  ereturn post `b' `V' ;

  ereturn local cmd     `"conleyreg"' ;
  ereturn local cmdline `"conleyreg `varlist'"' ;
  ereturn local coords  `"conley_coord1 conley_coord2"' ;
  ereturn local cutoffs `"conley_cutoff1 conley_cutoff2"' ;
  ereturn local depvar  `"`lhs_var'"' ;
  ereturn local model   `"conley ols"' ;
  ereturn local title   `"Conley OLS regression"' ;
  ereturn local vcetype `"Conley"' ;

  ereturn scalar N    = `N' ;
  ereturn scalar r2   = `r2' ;
  ereturn scalar r2_a = `r2_a' ;

  ereturn display ;
end ;

/* Define a wrapper program around "x_gmm.ado" that implements Conley's (1999) spatial GMM estimation */
capture program drop conleygmm ;
program define conleygmm, eclass ;
  version 14.2 ;
  syntax varlist [if], iv(varlist) ;

  tokenize `varlist' ;
  local lhs_var `1' ;
  local rhs_var `2' ;
  macro shift    2  ;
  local ctl_lst `*' ;

  local rhs_lst `"`rhs_var' `ctl_lst' _cons"' ;
  local reg_num : list sizeof rhs_lst ;
  local ins_lst `"`iv' `ctl_lst' _cons"' ;
  local ins_num : list sizeof ins_lst ;

  quietly { ;
    preserve ;
    keep `if' ;
    x_gmm conley_coord1 conley_coord2 conley_cutoff1 conley_cutoff2 `lhs_var' `rhs_var' `ctl_lst' conley_const `iv' `ctl_lst' conley_const, xreg(`reg_num') inst(`ins_num') coord(2) ;
    restore ;
  } ;

  tempname b V ;
  matrix `b' = betagmm' ;
  matrix `V' = cov_dep ;

  matrix drop J cov_dep betagmm bgmm1 cov2SLS b2SLS ;

  matrix colnames `b' = `rhs_lst' ;
  matrix rownames `b' = `lhs_var' ;
  matrix colnames `V' = `rhs_lst' ;
  matrix rownames `V' = `rhs_lst' ;

  qui count `if' ;
  local N = r(N) ;

  ereturn post `b' `V' ;

  ereturn local cmd       `"conleygmm"' ;
  ereturn local cmdline   `"conleygmm `varlist', iv(`iv')"' ;
  ereturn local coords    `"conley_coord1 conley_coord2"' ;
  ereturn local cutoffs   `"conley_cutoff1 conley_cutoff2"' ;
  ereturn local depvar    `"`lhs_var'"' ;
  ereturn local instd     `"`rhs_var'"' ;
  ereturn local insts     `"`ctl_lst' `iv'"' ;
  ereturn local title     `"Conley GMM regression"' ;
  ereturn local vcetype   `"Conley"' ;
  ereturn local estimator `"conley gmm"' ;
  ereturn local exogr     `"`ctl_lst'"' ;

  ereturn scalar N = `N' ;

  ereturn display ;
end ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/stab_freqy_conly.log", text replace ;

/*****************************************/
/* Load the AAGK cross-sectional dataset */
/*****************************************/

use "../../data/aagk_xcountry.dta", clear ;

/**************************************/
/* Apply relevant log transformations */
/**************************************/

gen lnnum_ccst_60_17_avg_pri1 = ln(1 + num_ccst_60_17_avg_pri1) ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable evercol_gbr "Ever a U.K. colony dummy" ;
label variable evercol_fra "Ever a French colony dummy" ;
label variable evercol_oth "Ever a non-U.K./non-French colony dummy" ;
label variable legor_uk    "British legal origin dummy" ;
label variable legor_fr    "French legal origin dummy" ;

/* Contemporary institutional variables */
label variable xconst_1960_2017_avg_p4_v17 "Executive constraints, 1960--2017 average" ;
label variable democ_1960_2017_shr_p4_v17  "Fraction of years under democracy, 1960--2017" ;
label variable autoc_1960_2017_shr_p4_v17  "Fraction of years under autocracy, 1960--2017" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet                 "Oil or gas reserve discovery" ;
label variable lnpop_1960_2017_avg_wdi          "Log population, 1960--2017 average" ;
label variable lngdppc_cu_usd_1960_2017_avg_wdi "Log GDP per capita, 1960--2017 average" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "evercol_gbr evercol_fra evercol_oth" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_1960_2017_avg_p4_v17" ;
global polregs "democ_1960_2017_shr_p4_v17 autoc_1960_2017_shr_p4_v17" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_1960_2017_avg_wdi" ;
global ypercap "lngdppc_cu_usd_1960_2017_avg_wdi" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;

/******************************************************/
/* Generate the sample indicators for the regressions */
/******************************************************/

qui egen smpl_flag1 = rowmiss(lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr}) ;
qui replace smpl_flag1 = (smpl_flag1 == 0) ;

qui egen smpl_flag2 = rowmiss(lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap}) ;
qui replace smpl_flag2 = (smpl_flag2 == 0) ;

/***********************/
/* Run the regressions */
/***********************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: conleyreg lnnum_ccst_60_17_avg_pri1 pdiv_aa if smpl_flag1 == 1 ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col1 ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: conleyreg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} if smpl_flag1 == 1 ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col2 ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: conleyreg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${contall} if smpl_flag1 == 1 ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col3 ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: conleyreg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${contall} if smpl_flag1 == 1 ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col4 ;

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: conleyreg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} if smpl_flag2 == 1 ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col5 ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: conleyreg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${contold} if smpl_flag1 == 1 & oldw == 1 ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col6 ;

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: conleyreg lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} if smpl_flag2 == 1 & oldw == 1 ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col7 ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: conleygmm lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${contall} if smpl_flag1 == 1, iv(mdist_addis) ;

/* -------- */
/* COLUMN 9 */
/* -------- */
eststo col9: conleygmm lnnum_ccst_60_17_avg_pri1 pdiv_aa ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} if smpl_flag2 == 1, iv(mdist_addis) ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 col9 using "../../results/tables/stab_freqy_conly.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(pdiv_aa, relax) 
  indicate("Continent dummies=${contall}" 
           "Controls for geography=${geovars}" 
           "Controls for ethnic diversity=${ethfrac} ${ethpolr}" 
           "Controls for institutions=${colhist} ${legalor} ${exconst} ${polregs}" 
           "Controls for oil, population, and income=${oilprod} ${popsize} ${ypercap}", labels("\$\times\$" " ")) 
  stats(N adjr2, fmt(%9.0f a2) labels("Observations" "Adjusted \$R^2\$") layout(@ @)) 
  varwidth(40) msign("\$-\$") label 
  prehead("\begin{tabular*}{610pt}{@{\extracolsep{\fill}}lccccccccccc}" 
          "\toprule" 
          "Cross-country sample:&\multicolumn{5}{c}{Global}&\multicolumn{2}{c}{Old World}&\multicolumn{2}{c}{Global}\\" 
          "\cmidrule(r){2-6}\cmidrule(lr){7-8}\cmidrule(l){9-10}") 
  numbers mlabels("OLS" "OLS" "OLS" "OLS" "OLS" "OLS" "OLS" "GMM" "GMM") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{9}{c}{Log number of new PRIO25 civil conflict onsets per year, 1960--2017}\\" 
           "\cmidrule{2-10}") 
  varlabels(, elist(pdiv_aa \addlinespace)) 
  prefoot("\midrule") postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
