/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/******************************************************************************************************************************************************************************/
/* TABLE: Population Diversity and the Incidence of Civil Conflict in Repeated Cross-Country Data - Robustness to Accounting for Alternative Correlates of Conflict Incidence */
/******************************************************************************************************************************************************************************/

version 14
# delimit ;
clear all ;

set more off ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/stab_incid_altco.log", text replace ;

/************************************/
/* Load the AAGK 5-yr panel dataset */
/************************************/

use "../../data/aagk_5yr_panel.dta", clear ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/*************************************************************************************/
/* Generate 1-period lags for the time-varying covariates and for conflict incidence */
/*************************************************************************************/

foreach var of varlist shrcol_gbr shrcol_fra shrcol_oth xconst_p4_v17 sdemoc_p4_v17 sautoc_p4_v17 lnpop_wdi lngdppc_cu_usd_wdi cconf_max_pri1 { ;
  qui gen `var'_l1 = l.`var' ;
} ;

/*************************/
/* Generate time dummies */
/*************************/

xi, noomit i.period ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable shrcol_gbr_l1 "Fraction of years as U.K. colony, lagged" ;
label variable shrcol_fra_l1 "Fraction of years as French colony, lagged" ;
label variable shrcol_oth_l1 "Fraction of years as other colony, lagged" ;
label variable legor_uk      "British legal origin dummy" ;
label variable legor_fr      "French legal origin dummy" ;

/* Proximate institutional variables */
label variable xconst_p4_v17_l1 "Average executive constraints, lagged" ;
label variable sdemoc_p4_v17_l1 "Fraction of years under democracy, lagged" ;
label variable sautoc_p4_v17_l1 "Fraction of years under autocracy, lagged" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet      "Oil or gas reserve discovery" ;
label variable lnpop_wdi_l1          "Log average population, lagged" ;
label variable lngdppc_cu_usd_wdi_l1 "Log average GDP per capita, lagged" ;

/* Lagged conflict incidence */
label variable cconf_max_pri1_l1 "Civil conflict incidence, lagged" ;

/* Alternative diversity variables */
label variable ef_fearon                       "Ethnic fractionalization (Fearon, 2003)" ;
label variable lfrac                           "Linguistic fractionalization (Alesina et al., 2003)" ;
label variable rfrac                           "Religious fractionalization (Alesina et al., 2003)" ;
label variable frac_fear_emr                   "Ethnolinguistic fractionalization (Esteban et al., 2012)" ;
label variable er_fear_delta005_emr            "Ethnolinguistic polarization (Esteban et al., 2012)" ;
label variable gini_fear_delta005_PERCAPTA_emr "Gini index of ethnolinguistic diversity (Esteban et al., 2012)" ;

/* Additional geographical variables */
label variable lmtnest_fl03 "Log percentage mountainous terrain" ;
label variable ncontig_fl03 "Noncontiguous state dummy" ;
label variable disease      "Disease richness" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethbase "efrac des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "shrcol_gbr_l1 shrcol_fra_l1 shrcol_oth_l1" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_p4_v17_l1" ;
global polregs "sdemoc_p4_v17_l1 sautoc_p4_v17_l1" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_wdi_l1" ;
global ypercap "lngdppc_cu_usd_wdi_l1" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;
global timedum "_Iperiod_30-_Iperiod_40" ;
global ethalt1 "ef_fearon des_pol15" ;
global ethalt2 "lfrac des_pol15" ;
global ethalt3 "rfrac des_pol15" ;
global ethalt4 "frac_fear_emr er_fear_delta005_emr gini_fear_delta005_PERCAPTA_emr" ;
global moregeo "lmtnest_fl03 ncontig_fl03 disease" ;

/***************************************************/
/* Generate a sample indicator for the regressions */
/***************************************************/

qui egen smpl_flag = rowmiss(cconf_max_pri1 pdiv_aa cconf_max_pri1_l1 ${geovars} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap}) ;
qui replace smpl_flag = (smpl_flag == 0 & period >= 30 & period <= 41) ;

/**********************************************/
/* Run the regressions for conflict incidence */
/**********************************************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: probit cconf_max_pri1 pdiv_aa cconf_max_pri1_l1 ${geovars} ${ethalt1} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timedum} if smpl_flag == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col1_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: probit cconf_max_pri1 pdiv_aa cconf_max_pri1_l1 ${geovars} ${ethalt2} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timedum} if smpl_flag == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col2_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: probit cconf_max_pri1 pdiv_aa cconf_max_pri1_l1 ${geovars} ${ethalt3} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timedum} if smpl_flag == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col3_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: probit cconf_max_pri1 pdiv_aa cconf_max_pri1_l1 ${geovars} ${ethalt4} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timedum} if smpl_flag == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col4_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: probit cconf_max_pri1 pdiv_aa cconf_max_pri1_l1 ${geovars} ${moregeo} ${ethbase} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timedum} if smpl_flag == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col5_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: ivprobit cconf_max_pri1 cconf_max_pri1_l1 ${geovars} ${ethalt1} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col6_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls cconf_max_pri1 cconf_max_pri1_l1 ${geovars} ${ethalt1} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col6 ;

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: ivprobit cconf_max_pri1 cconf_max_pri1_l1 ${geovars} ${ethalt2} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col7_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls cconf_max_pri1 cconf_max_pri1_l1 ${geovars} ${ethalt2} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col7 ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: ivprobit cconf_max_pri1 cconf_max_pri1_l1 ${geovars} ${ethalt3} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col8_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls cconf_max_pri1 cconf_max_pri1_l1 ${geovars} ${ethalt3} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col8 ;

/* -------- */
/* COLUMN 9 */
/* -------- */
eststo col9: ivprobit cconf_max_pri1 cconf_max_pri1_l1 ${geovars} ${ethalt4} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col9_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls cconf_max_pri1 cconf_max_pri1_l1 ${geovars} ${ethalt4} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col9 ;

/* --------- */
/* COLUMN 10 */
/* --------- */
eststo col10: ivprobit cconf_max_pri1 cconf_max_pri1_l1 ${geovars} ${moregeo} ${ethbase} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col10_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls cconf_max_pri1 cconf_max_pri1_l1 ${geovars} ${moregeo} ${ethbase} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col10 ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 col9 col10 using "../../results/tables/stab_incid_altco.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) drop(pdiv_aa:*, relax) keep(pdiv_aa ef_fearon lfrac rfrac ${ethalt4} ${moregeo}, relax) 
  order(pdiv_aa ef_fearon lfrac rfrac ${ethalt4} ${moregeo}) 
  indicate("Controls for all baseline covariates=${geovars} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} cconf_max_pri1_l1 _Iperiod_*", labels("\$\times\$" " ")) 
  stats(N N_clust r2_p, fmt(%9.0fc %9.0f a2) labels("Observations" "Countries" "Pseudo \$R^2\$") layout(@ @ @)) 
  varwidth(62) msign("\$-\$") label 
  prehead("\begin{tabular*}{805pt}{@{\extracolsep{\fill}}lcccccccccc}" 
          "\toprule" 
          "Cross-country sample:&\multicolumn{5}{c}{Old World}&\multicolumn{5}{c}{Global}\\" 
          "\cmidrule(r){2-6}\cmidrule(l){7-11}") 
  numbers mlabels("Probit" "Probit" "Probit" "Probit" "Probit" "IV Probit" "IV Probit" "IV Probit" "IV Probit" "IV Probit") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{10}{c}{Quinquennial PRIO25 civil conflict incidence, 1960--2017}\\" 
           "\cmidrule{2-11}") 
  eqlabels("", none) varlabels(,elist(disease \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1_me col2_me col3_me col4_me col5_me col6_me col7_me col8_me col9_me col10_me using "../../results/tables/stab_incid_altco.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) 
  varwidth(62) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(pdiv_aa "Marginal effect of diversity", elist(pdiv_aa \addlinespace)) ;

estout col1 col2 col3 col4 col5 col6 col7 col8 col9 col10 using "../../results/tables/stab_incid_altco.tex", style(tex) append 
  cells(none) 
  stats(fs_fsta, fmt(a2) labels("First-stage \$F\$ statistic") layout(@)) 
  varwidth(62) msign("\$-\$") 
  mlabels(none) collabels(none) 
  postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
