/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/**********************************************************************************************************************************************************************/
/* TABLE: Population Diversity and the Onset of Civil Conflict in Repeated Cross-Country Data - Robustness to Accounting for Alternative Correlates of Conflict Onset */
/**********************************************************************************************************************************************************************/

version 14
# delimit ;
clear all ;

set more off ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/stab_onset_altco.log", text replace ;

/************************************/
/* Load the AAGK 1-yr panel dataset */
/************************************/

use "../../data/aagk_1yr_panel.dta", clear ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/******************************************************************************************************/
/* Generate 1-year lags for the time-varying covariates and for conflict incidence and peace duration */
/******************************************************************************************************/

foreach var of varlist iscol_gbr iscol_fra iscol_oth xconst_p4_v17 ddemoc_p4_v17 dautoc_p4_v17 inst3_hs06 nwstate_hs06 lnpop_wdi lngdppc_cu_usd_wdi cconf_pri1 tap_a_pri1 { ;
  qui gen `var'_l1 = l.`var' ;
} ;

/**************************************************************************************************/
/* Generate cubic splines to control for the dependence of civil conflict onset on peace duration */
/**************************************************************************************************/

mkspline spline_pri1_  = tap_a_pri1_l1, cubic nknots(4) ;

/*************************/
/* Generate time dummies */
/*************************/

xi, noomit i.year ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable iscol_gbr_l1 "U.K. colony dummy, lagged" ;
label variable iscol_fra_l1 "French colony dummy, lagged" ;
label variable iscol_oth_l1 "Other colony dummy, lagged" ;
label variable legor_uk     "British legal origin dummy" ;
label variable legor_fr     "French legal origin dummy" ;

/* Proximate institutional variables */
label variable xconst_p4_v17_l1 "Executive constraints, lagged" ;
label variable ddemoc_p4_v17_l1 "Democracy dummy, lagged" ;
label variable dautoc_p4_v17_l1 "Autocracy dummy, lagged" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet      "Oil or gas reserve discovery" ;
label variable lnpop_wdi_l1          "Log population, lagged" ;
label variable lngdppc_cu_usd_wdi_l1 "Log GDP per capita, lagged" ;

/* Lagged conflict incidence */
label variable cconf_pri1_l1 "Civil conflict incidence, lagged" ;

/* Additional correlates of conflict onset */
label variable etdo4590_hs06   "Ethnic dominance" ;
label variable inst3_hs06_l1   "Political instability, lagged" ;
label variable nwstate_hs06_l1 "New state dummy, lagged" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "iscol_gbr_l1 iscol_fra_l1 iscol_oth_l1" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_p4_v17_l1" ;
global polregs "ddemoc_p4_v17_l1 dautoc_p4_v17_l1" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_wdi_l1" ;
global ypercap "lngdppc_cu_usd_wdi_l1" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;
global splines "spline_pri1_1 spline_pri1_2 spline_pri1_3" ;
global timedum "_Iyear_1960-_Iyear_1998" ;
global addcor1 "etdo4590_hs06 inst3_hs06_l1 nwstate_hs06_l1" ;
global addcor2 "etdo4590_hs06 inst3_hs06_l1" ;

/******************************************************/
/* Generate the sample indicators for the regressions */
/******************************************************/

qui egen smpl_flag1 = rowmiss(ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${addcor1} ${ethfrac} ${ethpolr}) ;
qui replace smpl_flag1 = (smpl_flag1 == 0 & year >= 1960 & year <= 1999) ;

qui egen smpl_flag2 = rowmiss(ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${addcor2} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap}) ;
qui replace smpl_flag2 = (smpl_flag2 == 0 & year >= 1960 & year <= 1999) ;

/******************************************/
/* Run the regressions for conflict onset */
/******************************************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: probit ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${contold} ${splines} ${timedum} if smpl_flag1 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col1_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: probit ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${addcor1} ${contold} ${splines} ${timedum} if smpl_flag1 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col2_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: probit ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${splines} ${timedum} if smpl_flag2 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col3_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: probit ccst_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${addcor2} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${splines} ${timedum} if smpl_flag2 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col4_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: ivprobit ccst_pri1 cconf_pri1_l1 ${geovars} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag1 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col5_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls ccst_pri1 cconf_pri1_l1 ${geovars} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col5 ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: ivprobit ccst_pri1 cconf_pri1_l1 ${geovars} ${addcor1} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag1 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col6_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls ccst_pri1 cconf_pri1_l1 ${geovars} ${addcor1} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col6 ;

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: ivprobit ccst_pri1 cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag2 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col7_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls ccst_pri1 cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col7 ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: ivprobit ccst_pri1 cconf_pri1_l1 ${geovars} ${addcor2} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag2 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col8_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls ccst_pri1 cconf_pri1_l1 ${geovars} ${addcor2} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${splines} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col8 ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 using "../../results/tables/stab_onset_altco.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) drop(pdiv_aa:*, relax) keep(pdiv_aa ${addcor1}, relax) order(pdiv_aa ${addcor1}) 
  indicate("Continent dummies=${contall}" 
           "Time dummies=_Iyear_*" 
           "Controls for temporal spillovers=cconf_pri1_l1 ${splines}" 
           "Controls for geography=${geovars}" 
           "Controls for ethnic diversity=${ethfrac} ${ethpolr}" 
           "Controls for institutions=${colhist} ${legalor} ${exconst} ${polregs}" 
           "Controls for oil, population, and income=${oilprod} ${popsize} ${ypercap}", labels("\$\times\$" " ")) 
  stats(N N_clust r2_p, fmt(%9.0fc %9.0f a2) labels("Observations" "Countries" "Pseudo \$R^2\$") layout(@ @ @)) 
  varwidth(40) msign("\$-\$") label 
  prehead("\begin{tabular*}{615pt}{@{\extracolsep{\fill}}lcccccccc}" 
          "\toprule" 
          "Cross-country sample:&\multicolumn{4}{c}{Old World}&\multicolumn{4}{c}{Global}\\" 
          "\cmidrule(r){2-5}\cmidrule(l){6-9}") 
  numbers mlabels("Probit" "Probit" "Probit" "Probit" "IV Probit" "IV Probit" "IV Probit" "IV Probit") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{8}{c}{Annual PRIO25 civil conflict onset, 1960--1999}\\" 
           "\cmidrule{2-9}") 
  eqlabels("", none) varlabels(,elist(nwstate_hs06_l1 \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1_me col2_me col3_me col4_me col5_me col6_me col7_me col8_me using "../../results/tables/stab_onset_altco.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) 
  varwidth(40) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(pdiv_aa "Marginal effect of diversity", elist(pdiv_aa \addlinespace)) ;

estout col1 col2 col3 col4 col5 col6 col7 col8 using "../../results/tables/stab_onset_altco.tex", style(tex) append 
  cells(none) 
  stats(fs_fsta, fmt(a2) labels("First-stage \$F\$ statistic") layout(@)) 
  varwidth(40) msign("\$-\$") 
  mlabels(none) collabels(none) 
  postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
