/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/*************************************************************************************************/
/* TABLE: Population Diversity and the Severity of Civil Conflict in Repeated Cross-Country Data */
/*************************************************************************************************/

# delimit ;
clear all ;

set more off ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/xtab_sever.log", text replace ;

/************************************/
/* Load the AAGK 5-yr panel dataset */
/************************************/

use "../../data/aagk_5yr_panel.dta", clear ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/*************************************************************************************/
/* Generate 1-period lags for the time-varying covariates and for conflict incidence */
/*************************************************************************************/

foreach var of varlist shrcol_gbr shrcol_fra shrcol_oth xconst_p4_v17 sdemoc_p4_v17 sautoc_p4_v17 lnpop_wdi lngdppc_cu_usd_wdi civtot_max_mepv17 domestic9_max_cnts18 { ;
  qui gen `var'_l1 = l.`var' ;
} ;

/*************************/
/* Generate time dummies */
/*************************/

xi, noomit i.period ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable shrcol_gbr_l1 "Fraction of years as U.K. colony, lagged" ;
label variable shrcol_fra_l1 "Fraction of years as French colony, lagged" ;
label variable shrcol_oth_l1 "Fraction of years as other colony, lagged" ;
label variable legor_uk      "British legal origin dummy" ;
label variable legor_fr      "French legal origin dummy" ;

/* Proximate institutional variables */
label variable xconst_p4_v17_l1 "Average executive constraints, lagged" ;
label variable sdemoc_p4_v17_l1 "Fraction of years under democracy, lagged" ;
label variable sautoc_p4_v17_l1 "Fraction of years under autocracy, lagged" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet      "Oil or gas reserve discovery" ;
label variable lnpop_wdi_l1          "Log average population, lagged" ;
label variable lngdppc_cu_usd_wdi_l1 "Log average GDP per capita, lagged" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "shrcol_gbr_l1 shrcol_fra_l1 shrcol_oth_l1" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_p4_v17_l1" ;
global polregs "sdemoc_p4_v17_l1 sautoc_p4_v17_l1" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_wdi_l1" ;
global ypercap "lngdppc_cu_usd_wdi_l1" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;
global timefe1 "_Iperiod_30-_Iperiod_40" ;
global timefe2 "_Iperiod_30-_Iperiod_39" ;

/******************************************************/
/* Generate the sample indicators for the regressions */
/******************************************************/

qui egen smpl_flag1a = rowmiss(civtot_max_mepv17 pdiv_aa civtot_max_mepv17_l1 ${geovars} ${ethfrac} ${ethpolr}) ;
qui replace smpl_flag1a = (smpl_flag1a == 0 & period >= 30 & period <= 41) ;

qui egen smpl_flag1b = rowmiss(civtot_max_mepv17 pdiv_aa civtot_max_mepv17_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap}) ;
qui replace smpl_flag1b = (smpl_flag1b == 0 & period >= 30 & period <= 41) ;

qui egen smpl_flag2a = rowmiss(domestic9_max_cnts18 pdiv_aa domestic9_max_cnts18_l1 ${geovars} ${ethfrac} ${ethpolr}) ;
qui replace smpl_flag2a = (smpl_flag2a == 0 & period >= 30 & period <= 40) ;

qui egen smpl_flag2b = rowmiss(domestic9_max_cnts18 pdiv_aa domestic9_max_cnts18_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap}) ;
qui replace smpl_flag2b = (smpl_flag2b == 0 & period >= 30 & period <= 40) ;

/*****************************************************************************************************************************/
/* Standardize the conflict severity variables in each of the regression samples since these variables are ordinal by nature */
/*****************************************************************************************************************************/

foreach var of varlist civtot_max_mepv17 civtot_max_mepv17_l1 { ;
  egen z1_`var' = std(`var') if smpl_flag1a == 1 & oldw == 1 ;
  egen z2_`var' = std(`var') if smpl_flag1b == 1 & oldw == 1 ;
  egen z3_`var' = std(`var') if smpl_flag1a == 1 ;
  egen z4_`var' = std(`var') if smpl_flag1b == 1 ;
} ;

/* Label the lagged standardized conflict severity variables */
label variable z1_civtot_max_mepv17_l1 "Civil conflict severity, lagged" ;
label variable z2_civtot_max_mepv17_l1 "Civil conflict severity, lagged" ;
label variable z3_civtot_max_mepv17_l1 "Civil conflict severity, lagged" ;
label variable z4_civtot_max_mepv17_l1 "Civil conflict severity, lagged" ;

foreach var of varlist domestic9_max_cnts18 domestic9_max_cnts18_l1 { ;
  egen z1_`var' = std(`var') if smpl_flag2a == 1 & oldw == 1 ;
  egen z2_`var' = std(`var') if smpl_flag2b == 1 & oldw == 1 ;
  egen z3_`var' = std(`var') if smpl_flag2a == 1 ;
  egen z4_`var' = std(`var') if smpl_flag2b == 1 ;
} ;

/* Label the lagged standardized conflict index variables */
label variable z1_domestic9_max_cnts18_l1 "Social conflict index, lagged" ;
label variable z2_domestic9_max_cnts18_l1 "Social conflict index, lagged" ;
label variable z3_domestic9_max_cnts18_l1 "Social conflict index, lagged" ;
label variable z4_domestic9_max_cnts18_l1 "Social conflict index, lagged" ;

/***************************************************/
/* Run the regressions for civil conflict severity */
/***************************************************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: reg z1_civtot_max_mepv17 pdiv_aa z1_civtot_max_mepv17_l1 ${geovars} ${contold} ${timefe1} if smpl_flag1a == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col1 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 41, percentiles(10 90) ;
eststo col1_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr z1_civtot_max_mepv17 pdiv_aa z1_civtot_max_mepv17_l1 ${geovars} ${contold} ${timefe1} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col1 ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: reg z2_civtot_max_mepv17 pdiv_aa z2_civtot_max_mepv17_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timefe1} if smpl_flag1b == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col2 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 41, percentiles(10 90) ;
eststo col2_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr z2_civtot_max_mepv17 pdiv_aa z2_civtot_max_mepv17_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timefe1} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col2 ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: ivregress 2sls z3_civtot_max_mepv17 z3_civtot_max_mepv17_l1 ${geovars} ${contall} ${timefe1} (pdiv_aa = mdist_addis) if smpl_flag1a == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the first-stage F-statistic */
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col3 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 41, percentiles(10 90) ;
eststo col3_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: ivregress 2sls z4_civtot_max_mepv17 z4_civtot_max_mepv17_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timefe1} (pdiv_aa = mdist_addis) if smpl_flag1b == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the first-stage F-statistic */
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col4 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 41, percentiles(10 90) ;
eststo col4_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/********************************************************/
/* Run the regressions for the index of social conflict */
/********************************************************/

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: reg z1_domestic9_max_cnts18 pdiv_aa z1_domestic9_max_cnts18_l1 ${geovars} ${contold} ${timefe2} if smpl_flag2a == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col5 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 40, percentiles(10 90) ;
eststo col5_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr z1_domestic9_max_cnts18 pdiv_aa z1_domestic9_max_cnts18_l1 ${geovars} ${contold} ${timefe2} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col5 ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: reg z2_domestic9_max_cnts18 pdiv_aa z2_domestic9_max_cnts18_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timefe2} if smpl_flag2b == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col6 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 40, percentiles(10 90) ;
eststo col6_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* Obtain the partial R2 of diversity */
pcorr z2_domestic9_max_cnts18 pdiv_aa z2_domestic9_max_cnts18_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timefe2} if e(sample) == 1 ;
matrix pc_mtrx = r(p_corr) ;
estadd scalar pc_pdiv = pc_mtrx[rownumb(pc_mtrx, "pdiv_aa"), 1]^2 : col6 ;

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: ivregress 2sls z3_domestic9_max_cnts18 z3_domestic9_max_cnts18_l1 ${geovars} ${contall} ${timefe2} (pdiv_aa = mdist_addis) if smpl_flag2a == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the first-stage F-statistic */
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col7 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 40, percentiles(10 90) ;
eststo col7_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: ivregress 2sls z4_domestic9_max_cnts18 z4_domestic9_max_cnts18_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timefe2} (pdiv_aa = mdist_addis) if smpl_flag2b == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the first-stage F-statistic */
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col8 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 40, percentiles(10 90) ;
eststo col8_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 using "../../results/tables/xtab_sever.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(pdiv_aa) 
  indicate("Continent dummies=${contall}" 
           "Time dummies=_Iperiod_*" 
           "Controls for temporal spillovers=z*_civtot_max_mepv17_l1 z*_domestic9_max_cnts18_l1" 
           "Controls for geography=${geovars}" 
           "Controls for ethnic diversity=${ethfrac} ${ethpolr}" 
           "Controls for institutions=${colhist} ${legalor} ${exconst} ${polregs}" 
           "Controls for oil, population, and income=${oilprod} ${popsize} ${ypercap}", labels("\$\times\$" " ")) 
  stats(N N_clust pc_pdiv adjr2, fmt(%9.0fc %9.0f a2 a2) labels("Observations" "Countries" "Partial \$R^2\$ of population diversity" "Adjusted \$R^2\$") layout(@ @ @ @)) 
  varwidth(44) msign("\$-\$") label 
  prehead("\begin{tabular*}{560pt}{@{\extracolsep{\fill}}lcccccccc}" 
          "\toprule" 
          "Cross-country sample:&\multicolumn{2}{c}{Old World}&\multicolumn{2}{c}{Global}&\multicolumn{2}{c}{Old World}&\multicolumn{2}{c}{Global}\\" 
          "\cmidrule(r){2-3}\cmidrule(lr){4-5}\cmidrule(lr){6-7}\cmidrule(l){8-9}") 
  numbers mlabels("OLS" "OLS" "2SLS" "2SLS" "OLS" "OLS" "2SLS" "2SLS") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{4}{c}{Quinquennial MEPV civil conflict}&\multicolumn{4}{c}{Quinquennial CNTS social conflict}\\" 
           "&\multicolumn{4}{c}{severity, 1960--2017}&\multicolumn{4}{c}{index, 1960--2014}\\" 
           "\cmidrule(r){2-5}\cmidrule(l){6-9}") 
  varlabels(,elist(pdiv_aa \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1_me col2_me col3_me col4_me col5_me col6_me col7_me col8_me using "../../results/tables/xtab_sever.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) 
  varwidth(44) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(me_1090 "Effect of 10th--90th \%ile move in diversity", elist(me_1090 \addlinespace)) ;

estout col1 col2 col3 col4 col5 col6 col7 col8 using "../../results/tables/xtab_sever.tex", style(tex) append 
  cells(none) 
  stats(fs_fsta, fmt(a2) labels("First-stage \$F\$ statistic") layout(@)) 
  varwidth(44) msign("\$-\$") 
  mlabels(none) collabels(none) 
  postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
