/*****************************************************************************/
/*** DESCRIPTION: REPLICATION CODE FOR "DIVERSITY AND CONFLICT"            ***/
/*** AUTHORS:     EREN ARBATLI, QUAMRUL ASHRAF, ODED GALOR, AND MARC KLEMP ***/
/*****************************************************************************/

/****************************************************************************************************************************************************************************************/
/* TABLE: Population Diversity and the Incidence of Civil Conflict in Repeated Cross-Country Data - Robustness to Examining the Annual Incidence or Quinquennial Prevalence of Conflict */
/****************************************************************************************************************************************************************************************/

version 14
# delimit ;
clear all ;

set more off ;

/*******************/
/* Open a log file */
/*******************/

capture log close ;
log using "../../results/logs/xtab_yrprv.log", text replace ;

/************************************/
/* Load the AAGK 1-yr panel dataset */
/************************************/

use "../../data/aagk_1yr_panel.dta", clear ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/***********************************************************************************/
/* Generate 1-year lags for the time-varying covariates and for conflict incidence */
/***********************************************************************************/

foreach var of varlist iscol_gbr iscol_fra iscol_oth xconst_p4_v17 ddemoc_p4_v17 dautoc_p4_v17 lnpop_wdi lngdppc_cu_usd_wdi cconf_pri1 { ;
  qui gen `var'_l1 = l.`var' ;
} ;

/*************************/
/* Generate time dummies */
/*************************/

xi, noomit i.year ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable iscol_gbr_l1 "U.K. colony dummy, lagged" ;
label variable iscol_fra_l1 "French colony dummy, lagged" ;
label variable iscol_oth_l1 "Other colony dummy, lagged" ;
label variable legor_uk     "British legal origin dummy" ;
label variable legor_fr     "French legal origin dummy" ;

/* Proximate institutional variables */
label variable xconst_p4_v17_l1 "Executive constraints, lagged" ;
label variable ddemoc_p4_v17_l1 "Democracy dummy, lagged" ;
label variable dautoc_p4_v17_l1 "Autocracy dummy, lagged" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet      "Oil or gas reserve discovery" ;
label variable lnpop_wdi_l1          "Log population, lagged" ;
label variable lngdppc_cu_usd_wdi_l1 "Log GDP per capita, lagged" ;

/* Lagged conflict incidence */
label variable cconf_pri1_l1 "Civil conflict incidence, lagged" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "iscol_gbr_l1 iscol_fra_l1 iscol_oth_l1" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_p4_v17_l1" ;
global polregs "ddemoc_p4_v17_l1 dautoc_p4_v17_l1" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_wdi_l1" ;
global ypercap "lngdppc_cu_usd_wdi_l1" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;
global timedum "_Iyear_1960-_Iyear_2016" ;

/******************************************************/
/* Generate the sample indicators for the regressions */
/******************************************************/

qui egen smpl_flag1 = rowmiss(cconf_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr}) ;
qui replace smpl_flag1 = (smpl_flag1 == 0 & year >= 1960 & year <= 2017) ;

qui egen smpl_flag2 = rowmiss(cconf_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap}) ;
qui replace smpl_flag2 = (smpl_flag2 == 0 & year >= 1960 & year <= 2017) ;

/*****************************************************/
/* Run the regressions for annual conflict incidence */
/*****************************************************/

/* -------- */
/* COLUMN 1 */
/* -------- */
eststo col1: probit cconf_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${contold} ${timedum} if smpl_flag1 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col1_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 2 */
/* -------- */
eststo col2: probit cconf_pri1 pdiv_aa cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timedum} if smpl_flag2 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col2_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* -------- */
/* COLUMN 3 */
/* -------- */
eststo col3: ivprobit cconf_pri1 cconf_pri1_l1 ${geovars} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag1 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col3_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls cconf_pri1 cconf_pri1_l1 ${geovars} ${contall} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col3 ;

/* -------- */
/* COLUMN 4 */
/* -------- */
eststo col4: ivprobit cconf_pri1 cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag2 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the average marginal effect of diversity */
eststo col4_me: margins, predict(pr) dydx(pdiv_aa) vce(unconditional) vsquish post ;

/* Obtain the first-stage F-statistic */
qui ivregress 2sls cconf_pri1 cconf_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if e(sample) == 1, vce(cluster cid) first noomitted vsquish ;
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col4 ;

/************************************/
/* Load the AAGK 5-yr panel dataset */
/************************************/

use "../../data/aagk_5yr_panel.dta", clear ;

/***********************************************************************************************/
/* Rescale those RHS variables whose regression coefficients would otherwise appear as "0.000" */
/***********************************************************************************************/

foreach var of varlist abslat ruggavg elevavg elevrng distcr { ;
  replace `var' = `var' / 1000 ;
} ;

replace mdist_addis = mdist_addis / 10 ;

/*************************************************************************************/
/* Generate 1-period lags for the time-varying covariates and for conflict incidence */
/*************************************************************************************/

foreach var of varlist shrcol_gbr shrcol_fra shrcol_oth xconst_p4_v17 sdemoc_p4_v17 sautoc_p4_v17 lnpop_wdi lngdppc_cu_usd_wdi cconf_shr_pri1 { ;
  qui gen `var'_l1 = l.`var' ;
} ;

/*************************/
/* Generate time dummies */
/*************************/

xi, noomit i.period ;

/****************************************************************************/
/* Assign appropriate labels to the RHS variables for the regression output */
/****************************************************************************/

/* Diversity variables */
label variable pdiv_aa   "Population diversity (ancestry adjusted)" ;
label variable efrac     "Ethnic fractionalization" ;
label variable des_pol15 "Ethnolinguistic polarization" ;

/* Geographical variables */
label variable abslat      "Absolute latitude" ;
label variable ruggavg     "Ruggedness" ;
label variable elevavg     "Mean elevation" ;
label variable elevrng     "Range of elevation" ;
label variable suitavg     "Mean land suitability" ;
label variable suitrng     "Range of land suitability" ;
label variable distcr      "Distance to nearest waterway" ;
label variable island      "Island nation dummy" ;
label variable mdist_addis "Migratory distance from East Africa (in 10,000 km)" ;

/* Deep institutional variables */
label variable shrcol_gbr_l1 "Fraction of years as U.K. colony, lagged" ;
label variable shrcol_fra_l1 "Fraction of years as French colony, lagged" ;
label variable shrcol_oth_l1 "Fraction of years as other colony, lagged" ;
label variable legor_uk      "British legal origin dummy" ;
label variable legor_fr      "French legal origin dummy" ;

/* Proximate institutional variables */
label variable xconst_p4_v17_l1 "Average executive constraints, lagged" ;
label variable sdemoc_p4_v17_l1 "Fraction of years under democracy, lagged" ;
label variable sautoc_p4_v17_l1 "Fraction of years under autocracy, lagged" ;

/* Oil, population, and income variables */
label variable anypetroleum_pet      "Oil or gas reserve discovery" ;
label variable lnpop_wdi_l1          "Log average population, lagged" ;
label variable lngdppc_cu_usd_wdi_l1 "Log average GDP per capita, lagged" ;

/* Lagged conflict prevalence */
label variable cconf_shr_pri1_l1 "Civil conflict prevalence, lagged" ;

/********************************************************/
/* Create global macro lists of the relevant covariates */
/********************************************************/

global ethfrac "efrac" ;
global ethpolr "des_pol15" ;
global geovars "abslat ruggavg elevavg elevrng suitavg suitrng distcr island" ;
global colhist "shrcol_gbr_l1 shrcol_fra_l1 shrcol_oth_l1" ;
global legalor "legor_uk legor_fr" ;
global exconst "xconst_p4_v17_l1" ;
global polregs "sdemoc_p4_v17_l1 sautoc_p4_v17_l1" ;
global oilprod "anypetroleum_pet" ;
global popsize "lnpop_wdi_l1" ;
global ypercap "lngdppc_cu_usd_wdi_l1" ;
global contold "africa asia" ;
global contall "africa asia namerica samerica oceania" ;
global timedum "_Iperiod_30-_Iperiod_40" ;

/******************************************************/
/* Generate the sample indicators for the regressions */
/******************************************************/

qui egen smpl_flag1 = rowmiss(cconf_shr_pri1 pdiv_aa cconf_shr_pri1_l1 ${geovars} ${ethfrac} ${ethpolr}) ;
qui replace smpl_flag1 = (smpl_flag1 == 0 & period >= 30 & period <= 41) ;

qui egen smpl_flag2 = rowmiss(cconf_shr_pri1 pdiv_aa cconf_shr_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap}) ;
qui replace smpl_flag2 = (smpl_flag2 == 0 & period >= 30 & period <= 41) ;

/************************************************************/
/* Run the regressions for quinquennial conflict prevalence */
/************************************************************/

/* -------- */
/* COLUMN 5 */
/* -------- */
eststo col5: reg cconf_shr_pri1 pdiv_aa cconf_shr_pri1_l1 ${geovars} ${contold} ${timedum} if smpl_flag1 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col5 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 41, percentiles(10 90) ;
eststo col5_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* -------- */
/* COLUMN 6 */
/* -------- */
eststo col6: reg cconf_shr_pri1 pdiv_aa cconf_shr_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contold} ${timedum} if smpl_flag2 == 1 & oldw == 1, vce(cluster cid) noomitted vsquish ;

/* Obtain the adjusted R2 */
estadd scalar adjr2 = e(r2_a) : col6 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 41, percentiles(10 90) ;
eststo col6_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* -------- */
/* COLUMN 7 */
/* -------- */
eststo col7: ivregress 2sls cconf_shr_pri1 cconf_shr_pri1_l1 ${geovars} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag1 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the first-stage F-statistic */
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col7 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 41, percentiles(10 90) ;
eststo col7_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/* -------- */
/* COLUMN 8 */
/* -------- */
eststo col8: ivregress 2sls cconf_shr_pri1 cconf_shr_pri1_l1 ${geovars} ${ethfrac} ${ethpolr} ${colhist} ${legalor} ${exconst} ${polregs} ${oilprod} ${popsize} ${ypercap} ${contall} ${timedum} (pdiv_aa = mdist_addis) if smpl_flag2 == 1, vce(cluster cid) first noomitted vsquish ;

/* Obtain the first-stage F-statistic */
estat firststage ;
matrix fs_mtrx = r(singleresults) ;
estadd scalar fs_fsta = fs_mtrx[1,4] : col8 ;

/* Obtain the effect of a 10th to 90th percentile move in diversity */
_pctile pdiv_aa if e(sample) == 1 & period == 41, percentiles(10 90) ;
eststo col8_me: nlcom (me_1090: _b[pdiv_aa] * (`r(r2)' - `r(r1)')), post ;

/**************************************/
/* Print the results to a LaTeX table */
/**************************************/

estout col1 col2 col3 col4 col5 col6 col7 col8 using "../../results/tables/xtab_yrprv.tex", style(tex) replace 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) drop(pdiv_aa:*, relax) keep(pdiv_aa) 
  indicate("Continent dummies=${contall}" 
           "Time dummies=_Iyear_* _Iperiod_*" 
           "Controls for temporal spillovers=cconf_pri1_l1 cconf_shr_pri1_l1" 
           "Controls for geography=${geovars}" 
           "Controls for ethnic diversity=${ethfrac} ${ethpolr}" 
           "Controls for institutions=${colhist} ${legalor} ${exconst} ${polregs}" 
           "Controls for oil, population, and income=${oilprod} ${popsize} ${ypercap}", labels("\$\times\$" " ")) 
  stats(N N_clust r2_p adjr2, fmt(%9.0fc %9.0f a2 a2) labels("Observations" "Countries" "Pseudo \$R^2\$" "Adjusted \$R^2\$") layout(@ @ @ @)) 
  varwidth(44) msign("\$-\$") label 
  prehead("\begin{tabular*}{595pt}{@{\extracolsep{\fill}}lcccccccc}" 
          "\toprule" 
          "Cross-country sample:&\multicolumn{2}{c}{Old World}&\multicolumn{2}{c}{Global}&\multicolumn{2}{c}{Old World}&\multicolumn{2}{c}{Global}\\" 
          "\cmidrule(r){2-3}\cmidrule(lr){4-5}\cmidrule(lr){6-7}\cmidrule(l){8-9}") 
  numbers mlabels("Probit" "Probit" "IV Probit" "IV Probit" "OLS" "OLS" "2SLS" "2SLS") collabels(none) 
  posthead("\midrule" 
           "&\multicolumn{4}{c}{Annual PRIO25 civil conflict}&\multicolumn{4}{c}{Quinquennial PRIO25 civil conflict}\\" 
           "&\multicolumn{4}{c}{incidence, 1960--2017}&\multicolumn{4}{c}{prevalence, 1960-2017}\\" 
           "\cmidrule(r){2-5}\cmidrule(l){6-9}") 
  eqlabels("", none) varlabels(,elist(pdiv_aa \addlinespace)) 
  prefoot("\midrule") postfoot("\addlinespace") ;

estout col1_me col2_me col3_me col4_me col5_me col6_me col7_me col8_me using "../../results/tables/xtab_yrprv.tex", style(tex) append 
  cells(b(nostar fmt(a2)) se(fmt(a2) par)) keep(pdiv_aa me_1090, relax) order(pdiv_aa me_1090) 
  varwidth(44) msign("\$-\$") 
  mlabels(none) collabels(none) 
  varlabels(pdiv_aa "Marginal effect of diversity" me_1090 "Effect of 10th--90th \%ile move in diversity", elist(me_1090 \addlinespace)) ;

estout col1 col2 col3 col4 col5 col6 col7 col8 using "../../results/tables/xtab_yrprv.tex", style(tex) append 
  cells(none) 
  stats(fs_fsta, fmt(a2) labels("First-stage \$F\$ statistic") layout(@)) 
  varwidth(44) msign("\$-\$") 
  mlabels(none) collabels(none) 
  postfoot("\bottomrule\addlinespace" "\end{tabular*}") ;

/***********************************************************/
/* Clean-up stored estimates, close the log file, and exit */
/***********************************************************/

est clear ;
log close ;

exit ;
